/* ########################################################################## */
/* (C) UPMC, 2010-2011                                                        */
/*     Authors:                                                               */
/*       Jean-Pierre Lozi <jean-pierre.lozi@lip6.fr>                          */
/*       Gaël Thomas <gael.thomas@lip6.fr>                                    */
/*       Florian David <florian.david@lip6.fr>                                */
/*       Julia Lawall <julia.lawall@lip6.fr>                                  */
/*       Gilles Muller <gilles.muller@lip6.fr>                                */
/* -------------------------------------------------------------------------- */
/* ########################################################################## */

/*
 * =============================================================================
 * Code based on Tudor David's libslock library.
 * =============================================================================
 */

#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <errno.h>
#include <sys/mman.h>

#include "liblock.h"
#include "liblock-fatal.h"

#include "extra-libslock-utils.h"
#include "extra-libslock-platform-defs.h"
#include "extra-libslock-atomic-ops.h"


typedef struct clh_qnode {
    volatile uint8_t locked;
    uint8_t padding[CACHE_LINE_SIZE - 1];
} clh_qnode;

typedef volatile clh_qnode *clh_qnode_ptr;
typedef clh_qnode_ptr clh_lock;

struct liblock_impl {
    unsigned int                       lock_id;
    pthread_mutex_t                    posix_lock;
    char                               pad1[CACHE_LINE_SIZE];
    clh_lock                          *lock;
    char __pad[pad_to_cache_line(sizeof(unsigned int) +
                                 sizeof(pthread_mutex_t) +
                                 sizeof(clh_lock))];
};

typedef struct clh_local_params {
    clh_qnode                         *my_qnode;
    clh_qnode                         *my_pred;
} clh_local_params;

__thread clh_local_params *local_params;
static volatile unsigned int cur_lock_number = 0;


static void lock_extra_clh(struct liblock_impl *impl)
{
    volatile clh_qnode *my_qnode = local_params[impl->lock_id].my_qnode;

    my_qnode->locked = 1;

    clh_qnode_ptr pred = (clh_qnode *)SWAP_PTR((volatile void *)impl->lock,
                                               (void *)my_qnode);

    if (pred == NULL)
    {
        local_params[impl->lock_id].my_pred = NULL;
        return;
    }

#if defined(OPTERON_OPTIMIZE)
    PREFETCHW(pred);
#endif  /* OPTERON_OPTIMIZE */

    while (pred->locked != 0)
    {
        PAUSE;

#if defined(OPTERON_OPTIMIZE)
        pause_rep(23);
        PREFETCHW(pred);
#endif  /* OPTERON_OPTIMIZE */
    }

    local_params[impl->lock_id].my_pred = (clh_qnode *)pred;
}

static void unlock_extra_clh(struct liblock_impl *impl)
{
    COMPILER_BARRIER;

    local_params[impl->lock_id].my_qnode->locked = 0;
    local_params[impl->lock_id].my_qnode =
        local_params[impl->lock_id].my_pred;
}

static struct liblock_impl *do_liblock_init_lock(extra_clh)
                               (liblock_lock_t *lock,
                                struct hw_thread *core,
                                pthread_mutexattr_t *attr)
{
    struct liblock_impl *impl =
        liblock_allocate(sizeof(struct liblock_impl));

    impl->lock_id = __sync_fetch_and_add(&cur_lock_number, 1);
    pthread_mutex_init(&impl->posix_lock, 0);

    impl->lock = (clh_lock*)liblock_allocate(sizeof(clh_lock));
    clh_qnode *a_node = (clh_qnode *)liblock_allocate(sizeof(clh_qnode));
    a_node->locked = 0;
    *(impl->lock) = a_node;

    MEM_BARRIER;

    return impl;
}

static int do_liblock_destroy_lock(extra_clh)(liblock_lock_t *lock)
{
    pthread_mutex_destroy(&lock->impl->posix_lock);

    free((clh_lock *)*(lock->impl->lock));
    free(lock->impl->lock);
    free(lock->impl);

    return 0;
}

static void* do_liblock_execute_operation(extra_clh)(liblock_lock_t *lock,
                                                    void* (*pending)(void*),
                                                    void *val)
{
    struct liblock_impl *impl = lock->impl;
    void *res;

    lock_extra_clh(impl);

    res = pending(val);

    unlock_extra_clh(impl);

    return res;
}

static void do_liblock_init_library(extra_clh)()
{}

static void do_liblock_kill_library(extra_clh)()
{}

static void do_liblock_run(extra_clh)(void (*callback)())
{
    if(__sync_val_compare_and_swap(&liblock_start_server_threads_by_hand,
                                   1, 0) != 1)
        fatal("servers are not managed by hand");
    if(callback)
        callback();
}

static int do_liblock_cond_init(extra_clh)(liblock_cond_t* cond)
{
    return cond->has_attr ?
        pthread_cond_init(&cond->impl.posix_cond, &cond->attr) :
        pthread_cond_init(&cond->impl.posix_cond, 0);
}

static int cond_timedwait(liblock_cond_t* cond,
                          liblock_lock_t* lock,
                          const struct timespec* ts)
{
    struct liblock_impl *impl = lock->impl;
    int res;

    pthread_mutex_lock(&impl->posix_lock);

    unlock_extra_clh(impl);

    if(ts)
        res = pthread_cond_timedwait(&cond->impl.posix_cond,
                                     &impl->posix_lock,
                                     ts);
    else
        res = pthread_cond_wait(&cond->impl.posix_cond, &impl->posix_lock);

    pthread_mutex_unlock(&impl->posix_lock);

    lock_extra_clh(impl);

    return res;
}

static int do_liblock_cond_timedwait(extra_clh)(liblock_cond_t* cond,
                                          liblock_lock_t* lock,
                                          const struct timespec* ts)
{
    return cond_timedwait(cond, lock, ts);
}

static int do_liblock_cond_wait(extra_clh)(liblock_cond_t* cond,
                                     liblock_lock_t* lock)
{
    return cond_timedwait(cond, lock, 0);
}

static int do_liblock_cond_signal(extra_clh)(liblock_cond_t* cond)
{
    return pthread_cond_signal(&cond->impl.posix_cond);
}

static int do_liblock_cond_broadcast(extra_clh)(liblock_cond_t* cond)
{
    return pthread_cond_broadcast(&cond->impl.posix_cond);
}

static int do_liblock_cond_destroy(extra_clh)(liblock_cond_t* cond)
{
    return pthread_cond_destroy(&cond->impl.posix_cond);
}

static void do_liblock_on_thread_start(extra_clh)(struct thread_descriptor* desc)
{
    int i;

    local_params = liblock_allocate(sizeof(clh_local_params) * MAX_LOCKS);

    for (i = 0; i < MAX_LOCKS; i++)
    {
        local_params[i].my_qnode = liblock_allocate(sizeof(clh_qnode));
        local_params[i].my_qnode->locked = 0;
        local_params[i].my_pred = NULL;
    }

    MEM_BARRIER;
}

static void do_liblock_on_thread_exit(extra_clh)(struct thread_descriptor* desc)
{
    // We do not free local data to avoid potential overhead in benchmarks.
/*
    int i;

    for (i = 0; i < MAX_LOCKS; i++)
    {
        free(local_params[i].my_qnode);
    }

    free(local_params);
*/
}

static void do_liblock_unlock_in_cs(extra_clh)(liblock_lock_t* lock)
{
    unlock_extra_clh(lock->impl);
}

static void do_liblock_relock_in_cs(extra_clh)(liblock_lock_t* lock)
{
    lock_extra_clh(lock->impl);
}

static void do_liblock_declare_server(extra_clh)(struct hw_thread* core)
{}

static void do_liblock_cleanup(extra_clh)(void)
{}

liblock_declare(extra_clh);

