/* ########################################################################## */
/* (C) UPMC, 2010-2011                                                        */
/*     Authors:                                                               */
/*       Jean-Pierre Lozi <jean-pierre.lozi@lip6.fr>                          */
/*       Gaël Thomas <gael.thomas@lip6.fr>                                    */
/*       Florian David <florian.david@lip6.fr>                                */
/*       Julia Lawall <julia.lawall@lip6.fr>                                  */
/*       Gilles Muller <gilles.muller@lip6.fr>                                */
/* -------------------------------------------------------------------------- */
/* ########################################################################## */

#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <errno.h>
#include <sys/mman.h>
#include <assert.h>

#include "liblock.h"
#include "liblock-fatal.h"

#include "extra-libslock-platform-defs.h"

#define MAX_NUMA_NODES       64
#define SLOTS                1024
//#define BATCH_LIMIT          1024
#define BATCH_LIMIT          2147483647

typedef struct tkt_lock { // ticket lock: TKT
    volatile int                 request;                    // next ticket
    volatile int                 grant;                      // now serving
    volatile int                 top_grant;                  // pass-locally handoff flag
    volatile int                 batch_count;                // tally of consecutive handoffs
    char __pad[pad_to_cache_line(4 * sizeof(int))];
} tkt_lock_t;

typedef struct pt_lock { // partitioned ticket lock: PTL
    volatile int                 request;
    volatile int                 grants[SLOTS];              // for performance, SLOTS should be >= # of NUMA nodes
    volatile int                 owner_ticket;
    char __pad[pad_to_cache_line(3 * sizeof(int))];
} pt_lock_t;

typedef struct cptltkt_lock {
    pt_lock_t                    top_lock;                   // top-level global lock
    tkt_lock_t        * volatile top_home;                   // local node with ownership of top-level lock
    tkt_lock_t                   local_lock[MAX_NUMA_NODES]; // set of node-level local locks: ensemble
    char __pad[pad_to_cache_line(sizeof(pt_lock_t) + sizeof(tkt_lock_t *) + sizeof(tkt_lock_t))];
} cptltkt_lock;

struct liblock_impl {
    unsigned int                 lock_id;
    pthread_mutex_t              posix_lock;
    char                         pad1[CACHE_LINE_SIZE];
    cptltkt_lock                 c;
    char __pad[pad_to_cache_line(sizeof(unsigned int) +
                                 sizeof(pthread_mutex_t) +
                                 sizeof(cptltkt_lock))];
};

static __thread int              my_node;

static void lock_extra_cohort(struct liblock_impl *impl)
{
    cptltkt_lock *c = &impl->c;

    tkt_lock_t *l = &c->local_lock[my_node];

    int t = __sync_fetch_and_add(&l->request, 1);

    while (l->grant != t) PAUSE();

    if (l->top_grant)
    {
        l->top_grant = 0;
        return;
    }

    t = __sync_fetch_and_add(&c->top_lock.request, 1);

    while (c->top_lock.grants[t % SLOTS] != t) PAUSE();

    c->top_lock.owner_ticket = t;
    c->top_home = l;
}

static void unlock_extra_cohort(struct liblock_impl *impl)
{
    cptltkt_lock *c = &impl->c;

    tkt_lock_t *l = c->top_home;

    int g = l->grant + 1;

    if (l->request != g)
    {
        if (--l->batch_count >= 0)
        {
            l->top_grant = 1;
            l->grant = g;
            return;
        }

        l->batch_count = BATCH_LIMIT;
    }

    int t = c->top_lock.owner_ticket + 1;
    c->top_lock.grants[t % SLOTS] = t;
    l->grant = g;
}

static struct liblock_impl *do_liblock_init_lock(extra_cohort)
                               (liblock_lock_t *lock,
                                struct hw_thread *core,
                                pthread_mutexattr_t *attr)
{
    struct liblock_impl *impl =
        liblock_allocate(sizeof(struct liblock_impl));

    pthread_mutex_init(&impl->posix_lock, 0);

    return impl;
}

static int do_liblock_destroy_lock(extra_cohort)(liblock_lock_t *lock)
{
    free(lock->impl);
    return 0;
}

static void* do_liblock_execute_operation(extra_cohort)(liblock_lock_t *lock,
                                                    void* (*pending)(void*),
                                                    void *val)
{
    struct liblock_impl *impl = lock->impl;
    void *res;

    lock_extra_cohort(impl);

    res = pending(val);

    unlock_extra_cohort(impl);

    return res;
}

static void do_liblock_init_library(extra_cohort)()
{}

static void do_liblock_kill_library(extra_cohort)()
{}

static void do_liblock_run(extra_cohort)(void (*callback)())
{
    if(__sync_val_compare_and_swap(&liblock_start_server_threads_by_hand,
                                   1, 0) != 1)
        fatal("servers are not managed by hand");
    if(callback)
        callback();
}

static int do_liblock_cond_init(extra_cohort)(liblock_cond_t* cond)
{
    return cond->has_attr ?
        pthread_cond_init(&cond->impl.posix_cond, &cond->attr) :
        pthread_cond_init(&cond->impl.posix_cond, 0);
}

static int cond_timedwait(liblock_cond_t* cond,
                          liblock_lock_t* lock,
                          const struct timespec* ts)
{
    struct liblock_impl *impl = lock->impl;
    int res;

    pthread_mutex_lock(&impl->posix_lock);

    unlock_extra_cohort(impl);

    if(ts)
        res = pthread_cond_timedwait(&cond->impl.posix_cond,
                                     &impl->posix_lock,
                                     ts);
    else
        res = pthread_cond_wait(&cond->impl.posix_cond, &impl->posix_lock);

    pthread_mutex_unlock(&impl->posix_lock);

    lock_extra_cohort(impl);

    return res;
}

static int do_liblock_cond_timedwait(extra_cohort)(liblock_cond_t* cond,
                                          liblock_lock_t* lock,
                                          const struct timespec* ts)
{
    return cond_timedwait(cond, lock, ts);
}

static int do_liblock_cond_wait(extra_cohort)(liblock_cond_t* cond,
                                     liblock_lock_t* lock)
{
    return cond_timedwait(cond, lock, 0);
}

static int do_liblock_cond_signal(extra_cohort)(liblock_cond_t* cond)
{
    return pthread_cond_signal(&cond->impl.posix_cond);
}

static int do_liblock_cond_broadcast(extra_cohort)(liblock_cond_t* cond)
{
    return pthread_cond_broadcast(&cond->impl.posix_cond);
}

static int do_liblock_cond_destroy(extra_cohort)(liblock_cond_t* cond)
{
    return pthread_cond_destroy(&cond->impl.posix_cond);
}

static void do_liblock_on_thread_start(extra_cohort)(struct thread_descriptor* desc)
{
    my_node = desc->id / CORES_PER_SOCKET;
    // printf("%d %d\n", getcpuid(), my_node);
}

static void do_liblock_on_thread_exit(extra_cohort)(struct thread_descriptor* desc)
{}

static void do_liblock_unlock_in_cs(extra_cohort)(liblock_lock_t* lock)
{}

static void do_liblock_relock_in_cs(extra_cohort)(liblock_lock_t* lock)
{}

static void do_liblock_declare_server(extra_cohort)(struct hw_thread* core)
{}

static void do_liblock_cleanup(extra_cohort)(void)
{}

liblock_declare(extra_cohort);

