/* ########################################################################## */
/* (C) UPMC, 2010-2011                                                        */
/*     Authors:                                                               */
/*       Jean-Pierre Lozi <jean-pierre.lozi@lip6.fr>                          */
/*       Gaël Thomas <gael.thomas@lip6.fr>                                    */
/*       Florian David <florian.david@lip6.fr>                                */
/*       Julia Lawall <julia.lawall@lip6.fr>                                  */
/*       Gilles Muller <gilles.muller@lip6.fr>                                */
/* -------------------------------------------------------------------------- */
/* ########################################################################## */
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <errno.h>
#include <sys/mman.h>
#include "liblock.h"
#include "liblock-fatal.h"

struct mcs_node {
    volatile struct mcs_node *volatile next;
    volatile int                       spin;
    char __pad[pad_to_cache_line(sizeof(void *) + sizeof(int))];
};

struct liblock_impl {
    unsigned int                       lock_id;
    pthread_mutex_t                    posix_lock;
    char                               pad[CACHE_LINE_SIZE];
    volatile struct mcs_node *volatile tail;
    char __pad[pad_to_cache_line(sizeof(unsigned int) +
                                 sizeof(pthread_mutex_t) +
                                 sizeof(void*))];
};

static inline void *xchg(struct mcs_node *volatile *ptr, struct mcs_node *x)
{
    asm volatile( "xchg %0,%1"
                  : "=r" (x)
                  : "m" (*ptr), "0" (x)
                  : "memory");

    return x;
}

static __thread struct mcs_node** my_nodes = NULL;

static volatile unsigned int cur_lock_number = 0;


static void lock_mcs(struct liblock_impl *impl)
{
    volatile struct mcs_node *tail, *me = my_nodes[impl->lock_id];

    me->next = 0;
    me->spin = 0;

    tail = __sync_lock_test_and_set(&impl->tail, me); //xchg(&impl->tail, me);

    /* No one there? */
    if (!tail)
        return;

    /* Someone there, need to link in */
    tail->next = me;

    /* Spin on my spin variable */
    while (!me->spin)
#ifdef WITH_YIELD
        YIELD();
#else
        PAUSE();
#endif

    return;
}

static void unlock_mcs(struct liblock_impl *impl)
{
    volatile struct mcs_node *me = my_nodes[impl->lock_id];

    /* No successor yet? */
    if (!me->next)
    {
        /* Try to atomically unlock */
        if (__sync_val_compare_and_swap(&impl->tail, me, 0) == me)
            return;

        /* Wait for successor to appear */
        while(!me->next)
            PAUSE();
    }

    /* Unlock next one */
    me->next->spin = 1;
}

static struct liblock_impl *do_liblock_init_lock(mcs)(liblock_lock_t *lock,
                                                      struct hw_thread *core,
                                                      pthread_mutexattr_t *attr)
{
    struct liblock_impl *impl =
        liblock_allocate(sizeof(struct liblock_impl));

    impl->tail = 0;
    impl->lock_id = __sync_fetch_and_add(&cur_lock_number, 1);
    pthread_mutex_init(&impl->posix_lock, 0);

    return impl;
}

static int do_liblock_destroy_lock(mcs)(liblock_lock_t *lock)
{
    pthread_mutex_destroy(&lock->impl->posix_lock);

    free(lock->impl);

    return 0;
}

static void* do_liblock_execute_operation(mcs)(liblock_lock_t *lock,
                                               void* (*pending)(void*),
                                               void *val)
{
    struct liblock_impl *impl = lock->impl;
    void *res;

    lock_mcs(impl);

    res = pending(val);

    unlock_mcs(impl);

    return res;
}

static void do_liblock_init_library(mcs)()
{}

static void do_liblock_kill_library(mcs)()
{}

static void do_liblock_run(mcs)(void (*callback)())
{
    if(__sync_val_compare_and_swap(&liblock_start_server_threads_by_hand,
                                   1, 0) != 1)
        fatal("servers are not managed by hand");
    if(callback)
        callback();
}

static int do_liblock_cond_init(mcs)(liblock_cond_t* cond)
{
    return cond->has_attr ?
        pthread_cond_init(&cond->impl.posix_cond, &cond->attr) :
        pthread_cond_init(&cond->impl.posix_cond, 0);
}

static int cond_timedwait(liblock_cond_t* cond,
                          liblock_lock_t* lock,
                          const struct timespec* ts)
{
    struct liblock_impl *impl = lock->impl;
    int res;

    pthread_mutex_lock(&impl->posix_lock);

    unlock_mcs(impl);

    if(ts)
        res = pthread_cond_timedwait(&cond->impl.posix_cond,
                                     &impl->posix_lock,
                                     ts);
    else
        res = pthread_cond_wait(&cond->impl.posix_cond, &impl->posix_lock);

    pthread_mutex_unlock(&impl->posix_lock);

    lock_mcs(impl);

    return res;
}

static int do_liblock_cond_timedwait(mcs)(liblock_cond_t* cond,
                                          liblock_lock_t* lock,
                                          const struct timespec* ts)
{
    return cond_timedwait(cond, lock, ts);
}

static int do_liblock_cond_wait(mcs)(liblock_cond_t* cond,
                                     liblock_lock_t* lock)
{
    return cond_timedwait(cond, lock, 0);
}

static int do_liblock_cond_signal(mcs)(liblock_cond_t* cond)
{
    return pthread_cond_signal(&cond->impl.posix_cond);
}

static int do_liblock_cond_broadcast(mcs)(liblock_cond_t* cond)
{
    return pthread_cond_broadcast(&cond->impl.posix_cond);
}

static int do_liblock_cond_destroy(mcs)(liblock_cond_t* cond)
{
    return pthread_cond_destroy(&cond->impl.posix_cond);
}

static void do_liblock_on_thread_start(mcs)(struct thread_descriptor* desc)
{
    int i;

    my_nodes = anon_mmap(r_align(sizeof(struct mcs_node *) * MAX_LOCKS,
                                 PAGE_SIZE));

    for (i = 0; i < MAX_LOCKS; i++)
    {
        my_nodes[i] = anon_mmap(r_align(sizeof(struct mcs_node), PAGE_SIZE));
    }
}

static void do_liblock_on_thread_exit(mcs)(struct thread_descriptor* desc)
{
    // We do not free local data to avoid potential overhead in benchmarks.
/*
    int i;

    for (i = 0; i < MAX_LOCKS; i++)
    {
        munmap((void *)(my_nodes[i]),
               r_align(sizeof(struct mcs_node), PAGE_SIZE));
    }

    munmap((void *)my_nodes,
           r_align(sizeof(struct mcs_node *) * MAX_LOCKS, PAGE_SIZE));
*/
}

static void do_liblock_unlock_in_cs(mcs)(liblock_lock_t* lock)
{
    unlock_mcs(lock->impl);
}

static void do_liblock_relock_in_cs(mcs)(liblock_lock_t* lock)
{
    lock_mcs(lock->impl);
}

static void do_liblock_declare_server(mcs)(struct hw_thread* core)
{}

static void do_liblock_cleanup(mcs)(void)
{}


liblock_declare(mcs);

