/* ########################################################################## */
/* (C) UPMC, 2010-2011                                                        */
/*     Authors:                                                               */
/*       Jean-Pierre Lozi <jean-pierre.lozi@lip6.fr>                          */
/*       Gaël Thomas <gael.thomas@lip6.fr>                                    */
/*       Florian David <florian.david@lip6.fr>                                */
/*       Julia Lawall <julia.lawall@lip6.fr>                                  */
/*       Gilles Muller <gilles.muller@lip6.fr>                                */
/* -------------------------------------------------------------------------- */
/* ########################################################################## */
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <errno.h>
#include <sys/mman.h>
#include "liblock-fatal.h"
#include "liblock.h"

struct mcs_node {
	struct mcs_node* volatile next;
	int              volatile spin;
	char __pad[0] __attribute__((aligned(64)));
};

struct liblock_lock {
	declare_liblock_lock_header();
	char                      pad0[pad_to_cache_line(sizeof(struct liblock_lock_header))];
	struct mcs_node* volatile tail;
	pthread_mutex_t           posix_lock;
};

static inline void *xchg(struct mcs_node* volatile* ptr, struct mcs_node* x) {
	__asm__ __volatile__(  "xchg %0,%1"
											 : "=r" (x)
											 : "m" (*ptr), "0" (x)
											 : "memory");

	return x;
}

__thread struct mcs_node __attribute__((aligned (CACHE_LINE_SIZE))) my_node;

static void lock_mcs(struct liblock_lock* lock) {
	struct mcs_node *tail, *me = &my_node;
	
	me->next = 0;
	me->spin = 0;

	tail = __sync_lock_test_and_set(&lock->tail, me); //xchg(&lock->tail, me);
	
	/* No one there? */
	if (!tail) 
		return;

	/* Someone there, need to link in */
	tail->next = me;

	__sync_synchronize();

	/* Spin on my spin variable */
	//int attempt = 0;
 	while (!me->spin) {
		//if(!(++attempt % 1000))
		//pthread_yield();
		//else
			PAUSE();
	}
	return;
}

static void unlock_mcs(struct liblock_lock* lock) {
	struct mcs_node* me = &my_node;
	/* No successor yet? */
	if (!me->next) {
		/* Try to atomically unlock */
		if (__sync_val_compare_and_swap(&lock->tail, me, 0) == me) 
			return;
	
		/* Wait for successor to appear */
		while(!me->next) 
			PAUSE();
	}

	/* Unlock next one */
	me->next->spin = 1;	
}

static struct liblock_lock* do_liblock_create_lock(mcs)(struct liblock* lib, struct core* core, pthread_mutexattr_t* attr) {
	struct liblock_lock* res = liblock_allocate(sizeof(struct liblock_lock));

	res->tail = 0;
	res->lib = lib;

	pthread_mutex_init(&res->posix_lock, 0);
	
	return res;
}

static void do_liblock_destroy_lock(mcs)(struct liblock_lock* lock) {
	free(lock);
}

static void* do_liblock_execute_operation(mcs)(struct liblock_lock* lock, void* (*pending)(void*), void* val) {
	void* res;
	//	static int n = 0; if(!(__sync_fetch_and_add(&n, 1) % 100)) printf("execute mcs %d\n", n);
	lock_mcs(lock);

	res = pending(val);

	unlock_mcs(lock);

	return res;
}

static void do_liblock_init_library(mcs)() {
}

static void do_liblock_kill_library(mcs)() {
}

static void do_liblock_run(mcs)(void (*callback)()) {
	if(__sync_val_compare_and_swap(&liblock_start_server_threads_by_hand, 1, 0) != 1)
		fatal("servers are not managed by hand");
	if(callback)
		callback();
}

static int do_liblock_cond_init(mcs)(liblock_cond_t* cond) { 
	fatal("implement me"); 
}

static int do_liblock_cond_wait(mcs)(liblock_cond_t* cond, struct liblock_lock* lock) { 
	fatal("implement me"); 
}

static int do_liblock_cond_timedwait(mcs)(liblock_cond_t* cond, struct liblock_lock* lock, const struct timespec* ts) { 
	fatal("implement me"); 
}

static int do_liblock_cond_signal(mcs)(liblock_cond_t* cond) { 
	fatal("implement me"); 
}

static int do_liblock_cond_broadcast(mcs)(liblock_cond_t* cond) { 
	fatal("implement me"); 
}

static int do_liblock_cond_destroy(mcs)(liblock_cond_t* cond) { 
	fatal("implement me"); 
}

static void do_liblock_on_thread_start(mcs)(struct thread_descriptor* desc) {
}

static void do_liblock_on_thread_exit(mcs)(struct thread_descriptor* desc) {
}

static void do_liblock_unlock_in_cs(mcs)(struct liblock_lock* lock) {
	unlock_mcs(lock);
}

static void do_liblock_relock_in_cs(mcs)(struct liblock_lock* lock) {
	lock_mcs(lock);
}

static void do_liblock_declare_server(mcs)(struct core* core) {
}

liblock_declare(mcs);

