/* ########################################################################## */
/* (C) UPMC, 2010-2011                                                        */
/*     Authors:                                                               */
/*       Jean-Pierre Lozi <jean-pierre.lozi@lip6.fr>                          */
/*       Gaël Thomas <gael.thomas@lip6.fr>                                    */
/*       Florian David <florian.david@lip6.fr>                                */
/*       Julia Lawall <julia.lawall@lip6.fr>                                  */
/*       Gilles Muller <gilles.muller@lip6.fr>                                */
/* -------------------------------------------------------------------------- */
/* ########################################################################## */
#include <string.h>
#include <errno.h>
#include <stdint.h>
#include "liblock-fatal.h"
#include "liblock.h"

struct qnode {
        volatile void *next;
        volatile char locked;
        char __pad[0] __attribute__((aligned(64)));
};

struct liblock_lock {
	declare_liblock_lock_header();
	char            pad0[pad_to_cache_line(sizeof(struct liblock_lock_header))];
	struct qnode *v __attribute__((aligned(64)));
};

static inline long xchg(long *ptr, long val)
{
        __asm__ volatile(
		"lock; xchgq %0, %1\n\t"
		: "+m" (*ptr), "+r" (val)
		:
		: "memory", "cc");
        return val;
}

static inline long cmpxchg(long *ptr, long old, long val)
{
    uint64_t out;
    __asm__ volatile(
    		"lock; cmpxchgq %2, %1"
		: "=a" (out), "+m" (*ptr)
		: "q" (val), "0"(old)
		: "memory");

    return out;
}

#define cpu_relax() asm volatile("rep; nop")

static inline void
arch_mcs_lock(struct liblock_lock *l, volatile struct qnode *mynode)
{
	struct qnode *predecessor;

	mynode->next = NULL;
	predecessor = (struct qnode *)xchg((long *)&l->v, (long)mynode);

	if (predecessor) {
		mynode->locked = 1;
		__sync_synchronize();
		predecessor->next = mynode;
		while (mynode->locked)
			cpu_relax();
	}
}

static inline void
arch_mcs_unlock(struct liblock_lock *l, volatile struct qnode *mynode)
{
	if (!mynode->next) {
		if (cmpxchg((long *)&l->v, (long)mynode, 0) == (long)mynode)
			return;
		while (!mynode->next)
			cpu_relax();
	}
	((struct qnode *)mynode->next)->locked = 0;
}

static struct liblock_lock* do_liblock_create_lock(mcsmit)(struct liblock* lib, struct core* server, pthread_mutexattr_t* attr) {
	struct liblock_lock* res = liblock_allocate(sizeof(struct liblock_lock));

	res->lib = lib;
	res->v = NULL;

	return res;
}

static void do_liblock_destroy_lock(mcsmit)(struct liblock_lock* lock) {
	free(lock);
}

__thread struct qnode __attribute__((aligned (CACHE_LINE_SIZE))) thread_node;

static void* do_liblock_execute_operation(mcsmit)(struct liblock_lock* lock, void* (*pending)(void*), void* val) {
	struct qnode* mynode = &thread_node;
	void* res;

	arch_mcs_lock(lock, mynode);
	
	res = pending(val);

	arch_mcs_unlock(lock, mynode);

	return res;
}

static void do_liblock_init_library(mcsmit)() {
}

static void do_liblock_kill_library(mcsmit)() {
}

static void do_liblock_run(mcsmit)(void (*callback)()) {
	if(__sync_val_compare_and_swap(&liblock_start_server_threads_by_hand, 1, 0) != 1)
		fatal("servers are not managed by hand");
	if(callback)
		callback();
}

static int do_liblock_cond_init(mcsmit)(liblock_cond_t* cond) { 
	fatal("implement me"); 
}

static int do_liblock_cond_wait(mcsmit)(liblock_cond_t* cond, struct liblock_lock* lock) { 
	fatal("implement me"); 
}

static int do_liblock_cond_timedwait(mcsmit)(liblock_cond_t* cond, struct liblock_lock* lock, const struct timespec* ts) { 
	fatal("implement me"); 
}

static int do_liblock_cond_signal(mcsmit)(liblock_cond_t* cond) { 
	fatal("implement me"); 
}

static int do_liblock_cond_broadcast(mcsmit)(liblock_cond_t* cond) { 
	fatal("implement me"); 
}

static int do_liblock_cond_destroy(mcsmit)(liblock_cond_t* cond) { 
	fatal("implement me"); 
}

static void do_liblock_on_thread_exit(mcsmit)(struct thread_descriptor* desc) {
}

static void do_liblock_on_thread_start(mcsmit)(struct thread_descriptor* desc) {
}

static void do_liblock_unlock_in_cs(mcsmit)(struct liblock_lock* lock) {
	fatal("implement me"); 
}

static void do_liblock_relock_in_cs(mcsmit)(struct liblock_lock* lock) {
	fatal("implement me"); 
}

static void do_liblock_declare_server(mcsmit)(struct core* core) {
}

liblock_declare(mcsmit);
