/* ########################################################################## */
/* (C) UPMC, 2010-2011                                                        */
/*     Authors:                                                               */
/*       Jean-Pierre Lozi <jean-pierre.lozi@lip6.fr>                          */
/*       Gaël Thomas <gael.thomas@lip6.fr>                                    */
/*       Florian David <florian.david@lip6.fr>                                */
/*       Julia Lawall <julia.lawall@lip6.fr>                                  */
/*       Gilles Muller <gilles.muller@lip6.fr>                                */
/* -------------------------------------------------------------------------- */
/* ########################################################################## */
#include <pthread.h>
#include <string.h>
#include <errno.h>
#include "liblock-fatal.h"
#include "liblock.h"

struct liblock_lock {
	declare_liblock_lock_header();
	char            pad0[pad_to_cache_line(sizeof(struct liblock_lock_header))];
	pthread_mutex_t mutex;
};

static struct liblock_lock* do_liblock_create_lock(posix)(struct liblock* lib, struct core* server, pthread_mutexattr_t* attr) {
	struct liblock_lock* res = liblock_allocate(sizeof(struct liblock_lock));

	pthread_mutex_init(&res->mutex, attr);
	res->lib = lib;

	//printf("create posix lock %p\n", res);

	return res;
}

static void do_liblock_destroy_lock(posix)(struct liblock_lock* lock) {
	free(lock);
}

static void* do_liblock_execute_operation(posix)(struct liblock_lock* lock, void* (*pending)(void*), void* val) {
	void* res;
//	printf("posix locking\n");
	pthread_mutex_lock(&lock->mutex);
	res = pending(val);
	pthread_mutex_unlock(&lock->mutex);
//	printf("unlocking\n");
	
	return res;
}

static void do_liblock_init_library(posix)() {
}

static void do_liblock_kill_library(posix)() {
}

static void do_liblock_run(posix)(void (*callback)()) {
	if(__sync_val_compare_and_swap(&liblock_start_server_threads_by_hand, 1, 0) != 1)
		fatal("servers are not managed by hand");
	if(callback)
		callback();
}

static int do_liblock_cond_init(posix)(liblock_cond_t* cond) { 
	return cond->has_attr ?
		pthread_cond_init(&cond->posix_cond, &cond->attr) :
		pthread_cond_init(&cond->posix_cond, 0);
}

static int do_liblock_cond_wait(posix)(liblock_cond_t* cond, struct liblock_lock* lock) { 
	return pthread_cond_wait(&cond->posix_cond, &lock->mutex);
}

static int do_liblock_cond_timedwait(posix)(liblock_cond_t* cond, struct liblock_lock* lock, const struct timespec* ts) { 
	return pthread_cond_timedwait(&cond->posix_cond, &lock->mutex, ts);
}

static int do_liblock_cond_signal(posix)(liblock_cond_t* cond) { 
	return pthread_cond_signal(&cond->posix_cond);
}

static int do_liblock_cond_broadcast(posix)(liblock_cond_t* cond) { 
	return pthread_cond_broadcast(&cond->posix_cond);
}

static int do_liblock_cond_destroy(posix)(liblock_cond_t* cond) { 
	fatal("implement me"); 
}

static void do_liblock_on_thread_start(posix)(struct thread_descriptor* desc) {
}

static void do_liblock_on_thread_exit(posix)(struct thread_descriptor* desc) {
}

static void do_liblock_unlock_in_cs(posix)(struct liblock_lock* lock) {
	pthread_mutex_unlock(&lock->mutex);
}

static void do_liblock_relock_in_cs(posix)(struct liblock_lock* lock) {
	pthread_mutex_lock(&lock->mutex);
}

static void do_liblock_declare_server(posix)(struct core* core) {
}

liblock_declare(posix);
